<?php
/**
 * Module Name: AI
 * Description: Module correspond for all AI related functionality like text generation, translation, etc.
 *
 * @since 7.7
 */

if ( ! defined( 'ABSPATH' ) ) {
	die( '-1' );
}

require_once vc_manager()->path( 'MODULES_DIR', 'ai/class-vc-ai-module-settings.php' );

/**
 * Module entry point.
 *
 * @since 7.7
 */
class Vc_Ai_Module {

	/**
	 * Settings object.
	 *
	 * @var Vc_Ai_Module_Settings
	 */
	public $settings;

	/**
	 * Vc_Ai_Module constructor.
	 *
	 * @since 8.0
	 */
	public function __construct() {
		$this->settings = new Vc_Ai_Module_Settings();
		$this->settings->init();
	}

	/**
	 * Module init.
	 *
	 * @since 7.7
	 */
	public function init() {
		require_once vc_path_dir( 'MODULES_DIR', 'ai/helpers.php' );

		add_action( 'wp_ajax_wpb_ai_api_get_response', [ $this, 'get_ai_api_response' ] );
		add_action( 'wp_ajax_wpb_ai_generate_content_check_cache', [ $this, 'get_generate_ai_content_check_cache' ] );
		add_action( 'wp_ajax_wpb_ai_get_modal_data', [ $this, 'get_modal_ai_data' ] );
		add_action( 'wp_ajax_wpb_ai_get_token_usage', [ $this, 'get_token_usage_data' ] );

		add_filter( 'vc_single_param_edit_holder_output', [ $this, 'add_ai_icon_to_attributes' ], 10, 2 );

		add_filter( 'vc_roles_parts_list', [ $this, 'add_ai_role_parts' ], 10, 1 );

		add_filter( 'vc_get_editor_locale', [ $this, 'add_module_localization' ], 10, 1 );
		add_filter( 'vc_get_settings_locale', [ $this, 'add_module_localization' ], 10, 1 );
	}

	/**
	 * Get response from AI API.
	 *
	 * @sine 7.2
	 */
	public function get_ai_api_response() {
		vc_user_access()->checkAdminNonce()->validateDie();

		require_once vc_path_dir( 'MODULES_DIR', 'ai/class-vc-ai-api-connector.php' );
		$ai_api_connector = new Vc_Ai_Api_Connector();

		$content = $ai_api_connector->get_ai_content( vc_request_param( 'data' ) );

		if ( is_wp_error( $content ) ) {
			wp_send_json_error( $content );
		}

		wp_send_json_success( $content );
	}

	/**
	 * Get content generated by AI.
	 *
	 * @sine 7.2
	 */
	public function get_generate_ai_content_check_cache() {
		vc_user_access()->checkAdminNonce()->validateDie();

		require_once vc_path_dir( 'MODULES_DIR', 'ai/class-vc-ai-api-connector.php' );
		$api_connector = new Vc_Ai_Api_Connector();

		$content = $api_connector->get_api_response_data_from_cache( vc_request_param( 'data' ) );
		if ( is_wp_error( $content ) ) {
			wp_send_json_error( $content );
		}

		wp_send_json_success( $content );
	}

	/**
	 * Get AI modal content.
	 *
	 * @sine 7.2
	 */
	public function get_modal_ai_data() {
		vc_user_access()->checkAdminNonce()->validateDie();

		require_once vc_path_dir( 'MODULES_DIR', 'ai/class-vc-ai-modal-controller.php' );
		$modal_controller = new Vc_Ai_Modal_Controller();

		$data = $modal_controller->get_modal_data( vc_request_param( 'data' ) );

		// in case of error we should show it inside modal content.
		wp_send_json_success( $data );
	}

	/**
	 * Get AI token usage corresponding to current active license.
	 *
	 * @sine 7.7
	 */
	public function get_token_usage_data() {
		vc_user_access()->checkAdminNonce()->validateDie();

		require_once vc_path_dir( 'MODULES_DIR', 'ai/class-vc-ai-modal-controller.php' );
		$modal_controller = new Vc_Ai_Modal_Controller();

		$data = $modal_controller->get_token_usage_request();
		if ( is_wp_error( $data ) ) {
			wp_send_json_error( $data );
		}

		wp_send_json_success( $data );
	}

	/**
	 * Add ai icon to our element attributes.
	 *
	 * @since 7.7
	 * @param string $output
	 * @param array $param
	 * @return string
	 */
	public function add_ai_icon_to_attributes( $output, $param ) {
		$form_line = '<div class="edit_form_line">';
		$pos = strpos( $output, $form_line );
		if ( false === $pos ) {
			return $output;
		}

		$add_icon = $this->render_ai_icon( $param );

		return substr_replace( $output, $add_icon, $pos + strlen( $form_line ), 0 );
	}

	/**
	 * Generate html for AI icon.
	 *
	 * @see $this->get_lib_ai_icon_words to find a list of words
	 * if element name contain than we show AI icon for it
	 *
	 * @see $this->get_ai_param_types to find a list of element types
	 * that has AI functionality
	 *
	 * @param array $param
	 * @return string
	 * @since 7.2
	 */
	public function render_ai_icon( $param ) {
		$ai_icon = '';

		if ( ! $this->is_user_has_access_to_ai( $param['type'] ) ) {
			return $ai_icon;
		}

		$ai_param_types = $this->get_ai_param_types();
		if ( empty( $param['heading'] ) || ! is_array( $ai_param_types ) ) {
			return $ai_icon;
		}

		$is_ai_acceptable_param = in_array( $param['type'], $ai_param_types );
		$is_output_ai_icon = $is_ai_acceptable_param || $this->is_available_ai_text_field( $param );

		if ( $is_output_ai_icon ) {
			$ai_icon = wpb_get_ai_icon_template( $param['type'], $this->get_field_id( $param ), false );
		}

		return $ai_icon;
	}

	/**
	 * Get AI icon template with field id arranged
	 *
	 * @param array $param
	 * @return string
	 */
	public function get_field_id( $param ) {
		$field_id = strtolower( preg_replace( '/[^A-Za-z0-9]+/', '_', $param['heading'] ) );
		return $param['type'] . '_' . $field_id;
	}

	/**
	 * Fot a text_filed param type we restrict the AI icon to be shown
	 * only for specific words in the heading.
	 *
	 * @since 8.3
	 * @param array $param
	 * @return bool
	 */
	public function is_available_ai_text_field( $param ) {
		$heading = $param['heading'];
		$heading_words = preg_split( '/[\s,]+/', $heading );
		$is_content = false;
		foreach ( $heading_words as $word ) {
			$word = strtolower( $word );
			$lib_of_words = $this->get_lib_ai_icon_words();
			if ( is_array( $lib_of_words ) && in_array( $word, $lib_of_words ) ) {
				$is_content = true;
			}
		}

		return 'textfield' === $param['type'] && 'el_class' !== $param['param_name'] && $is_content;
	}

	/**
	 * Get list of words that text_filed type element name should
	 * have to apply AI functionality to that.
	 *
	 * @since 7.2
	 * @return array
	 */
	public function get_lib_ai_icon_words() {
		$word_list = [
			'label',
			'title',
			'text',
			'content',
			'description',
			'message',
			'heading',
			'subheading',
		];

		return apply_filters( 'wpb_module_ai_text_field_words', $word_list );
	}

	/**
	 * Get list of param types that has AI functionality
	 *
	 * @since 7.2
	 * @return array
	 */
	public function get_ai_param_types() {
		$params = [
			'textarea_html',
			'textarea',
			'textarea_raw_html',
			'textarea_ace',
		];
		$params_addons = [
			'uc_textfield',
			'uc_textarea',
			'uc_editor',
			'us_textarea',
			'us_text',
		];

		return apply_filters( 'wpb_module_ai_element_param_types', array_merge( $params, $params_addons ) );
	}

	/**
	 * Check if user has permission to AI
	 *
	 * @param string $type
	 *
	 * @return bool
	 * @since 7.2
	 */
	public function is_user_has_access_to_ai( $type ) {
		$access_part = ( 'textarea_raw_html' === $type ) ? 'code_ai' : 'text_ai';
		return vc_user_access()->part( $access_part )->can()->get();
	}

	/**
	 * Add AI role parts to our roles manager.
	 *
	 * @since 7.7
	 * @param array $parts
	 * @return array
	 */
	public function add_ai_role_parts( $parts ) {
		$parts[] = 'code_ai';
		$parts[] = 'text_ai';

		return $parts;
	}

	/**
	 * Add module localization.
	 *
	 * @since 7.7
	 * @param array $localization
	 * @return array
	 */
	public function add_module_localization( $localization ) {
		$localization['ai_credit_usage'] = esc_html__( 'Credit usage: ', 'js_composer' );
		$localization['ai_response_error'] = esc_html__(
			'An error occurred when requesting a response from WPBakery AI (Code: 623)',
			'js_composer'
		);

		return $localization;
	}
}
